<?php

declare(strict_types=1);

namespace CGA\Sync\Traits;

use CGA\Sync\Facades\Sync;
use CGA\Sync\Models\SyncDocument;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\MorphOne;
use Illuminate\Database\Eloquent\Relations\Pivot;
use Illuminate\Support\Str;
use Ramsey\Uuid\UuidInterface;

/**
 * @mixin Model
 *
 * @phpstan-ignore trait.unused
 */
trait Syncable
{
    use SyncsBiDirectionally;

    /** @return MorphOne<SyncDocument, $this> */
    public function syncDocument(): MorphOne
    {
        return $this->morphOne(Sync::getDocumentModel(), 'syncable');
    }

    public function getSyncDocument(): ?SyncDocument
    {
        return $this->syncDocument;
    }

    public function getSyncRevisionId(): ?string
    {
        return $this->syncDocument?->revision_id;
    }

    /**
     * Scope to find entities by their sync ID
     */
    public function scopeBySyncId(Builder $query, string|UuidInterface $syncId): Builder
    {
        return $query->whereHas('syncDocument', function (Builder $q) use ($syncId) {
            $q->where('sync_id', $syncId);
        });
    }

    public function getSyncId(): string
    {
        if ($this->relationLoaded('syncDocument')) {
            return $this->syncDocument->sync_id;
        }

        return $this->syncDocument()->first()->sync_id ?? (string) Str::orderedUuid();
    }

    /**
     * Check if this model is a Pivot model.
     */
    public function isPivotModel(): bool
    {
        return $this instanceof Pivot;
    }

    /**
     * Validate that this model meets sync requirements.
     */
    public function validateSyncRequirements(): void
    {
        if ($this->isPivotModel()) {
            if (! $this->getKeyName()) {
                throw new \InvalidArgumentException('Syncable Pivot models must have a primary key defined');
            }

            if (! $this->incrementing) {
                throw new \InvalidArgumentException('Syncable Pivot models must have $incrementing = true to ensure proper primary key generation');
            }
        }
    }
}
