<?php

declare(strict_types=1);

namespace CGA\Sync\Services;

use CGA\Sync\Enums\Status;
use CGA\Sync\Exceptions\SyncConflictException;
use CGA\Sync\Facades\Sync;
use CGA\Sync\Models\SyncDocument;
use Exception;
use Illuminate\Support\Collection;

class DocumentProcessor
{
    /**
     * Process a sync document with conflict detection
     *
     * @throws SyncConflictException
     */
    public function process(
        string $syncId,
        Collection $document,
        Collection $clientDetails,
        ?string $expectedRevisionId = null,
        ?bool $deleted = false
    ): SyncDocument {
        // Check for revision conflicts
        $existingDocument = Sync::getDocumentModel()::where('sync_id', $syncId)->first();

        if ($expectedRevisionId) {
            if ($existingDocument && $existingDocument->revision_id !== $expectedRevisionId) {
                throw SyncConflictException::revisionMismatch(
                    syncId: $syncId,
                    conflictingModel: $existingDocument->syncable,
                );
            }
        }

        $syncableType = $existingDocument->syncable_type ?? $document->get('syncable_type');

        if (! $syncableType) {
            throw new Exception('Syncable type is required');
        }

        return Sync::upsertSyncDocument($syncId, [
            'document' => $document,
            'client_details' => $clientDetails,
            'syncable_type' => $syncableType,
            'status' => Status::PENDING,
            'deleted' => $deleted,
        ]);
    }

    /**
     * Find a sync document by ID
     */
    public function findDocument(string $syncId): SyncDocument
    {
        return Sync::getDocumentModel()::where('sync_id', $syncId)->firstOrFail();
    }
}
