<?php

declare(strict_types=1);

namespace CGA\Sync\Traits;

use CGA\Sync\Contracts\SyncableContract;
use CGA\Sync\Events\SyncableModelCreated;
use CGA\Sync\Events\SyncableModelDeleted;
use CGA\Sync\Events\SyncableModelRestored;
use CGA\Sync\Events\SyncableModelUpdated;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

/**
 * @mixin Model
 *
 * @phpstan-ignore trait.unused
 */
trait SyncsBiDirectionally
{
    protected static bool $skipSyncEvents = false;

    /**
     * Boot the Syncable trait for a model.
     * Registers model event listeners for bi-directional sync.
     */
    public static function bootSyncable(): void
    {
        static::created(fn(Model&SyncableContract $model) => ! static::$skipSyncEvents && SyncableModelCreated::dispatch($model));
        static::updated(fn(Model&SyncableContract $model) => ! static::$skipSyncEvents && SyncableModelUpdated::dispatch($model, $model->getOriginal()));
        static::deleted(fn(Model&SyncableContract $model) => ! static::$skipSyncEvents && SyncableModelDeleted::dispatch($model, $model->getOriginal()));

        if (in_array(SoftDeletes::class, class_uses_recursive(static::class))) {
            static::restored(fn(Model&SyncableContract $model) => ! static::$skipSyncEvents && SyncableModelRestored::dispatch($model));
        }
    }

    public static function withoutSyncEvents(callable $callback): mixed
    {
        static::$skipSyncEvents = true;

        try {
            return $callback();
        } finally {
            static::$skipSyncEvents = false;
        }
    }

    public function skipSyncEvents(callable $callback): mixed
    {
        return static::withoutSyncEvents($callback);
    }
}
