<?php

declare(strict_types=1);

namespace CGA\Sync\Schema;

use CGA\Sync\Contracts\SchemaHandler;
use CGA\Sync\Schema\Handlers\EnumSchemaHandler;
use Illuminate\Database\Eloquent\Model;

class SchemaResolver
{
    /**
     * @var array<SchemaHandler>
     */
    protected array $handlers = [];

    public function __construct()
    {
        $this->loadHandlers();
    }

    /**
     * Resolve the complete schema for a given model.
     *
     * @return array<string, array<string, mixed>>
     */
    public function resolve(Model $model): array
    {
        $schema = [];

        // Get all attributes from the model
        // We'll check casts as those are the ones we care about for schema
        foreach (array_keys($model->getCasts()) as $attribute) {
            foreach ($this->handlers as $handler) {
                if ($handler->supports($attribute, $model)) {
                    $schema[$attribute] = $handler->resolve($attribute, $model);
                    break; // First match wins
                }
            }
        }

        return $schema;
    }

    /**
     * Load handlers from configuration.
     */
    protected function loadHandlers(): void
    {
        $handlerClasses = config('sync.schema.handlers', [
            EnumSchemaHandler::class,
        ]);

        foreach ($handlerClasses as $handlerClass) {
            $this->handlers[] = app($handlerClass);
        }
    }

    /**
     * Get the loaded handlers.
     *
     * @return array<SchemaHandler>
     */
    public function getHandlers(): array
    {
        return $this->handlers;
    }
}
