<?php

declare(strict_types=1);

namespace CGA\Sync\Schema\Handlers;

use BackedEnum;
use CGA\Sync\Contracts\SchemaHandler;
use Illuminate\Database\Eloquent\Model;
use UnitEnum;

class EnumSchemaHandler implements SchemaHandler
{
    public function supports(string $attribute, Model $model): bool
    {
        $cast = $model->getCasts()[$attribute] ?? null;

        if (! $cast) {
            return false;
        }

        return is_subclass_of($cast, BackedEnum::class) || is_subclass_of($cast, UnitEnum::class);
    }

    public function resolve(string $attribute, Model $model): array
    {
        $enumClass = $model->getCasts()[$attribute];
        $cases = $enumClass::cases();
        $locales = $this->getLocales();
        $fallbackLocale = $this->getFallbackLocale();
        $originalLocale = app()->getLocale();

        $options = [];

        foreach ($cases as $case) {
            $labels = [];

            foreach ($locales as $locale) {
                app()->setLocale($locale);

                $label = $this->getLabelForCase($case);

                // If translation is missing (returns key), use fallback locale
                if ($this->isTranslationMissing($label, $case)) {
                    app()->setLocale($fallbackLocale);
                    $label = $this->getLabelForCase($case);
                    app()->setLocale($locale);
                }

                $labels[$locale] = $label;
            }

            $options[] = [
                'value' => $case instanceof BackedEnum ? $case->value : $case->name,
                'labels' => $labels,
            ];
        }

        // Restore original locale
        app()->setLocale($originalLocale);

        return [
            'type' => 'enum',
            'options' => $options,
        ];
    }

    protected function getLabelForCase(UnitEnum $case): string
    {
        // Check for description() method first
        if (method_exists($case, 'description')) {
            return $case->description();
        }

        // Check for label() method
        if (method_exists($case, 'label')) {
            return $case->label();
        }

        // Check for getLabel() method
        if (method_exists($case, 'getLabel')) {
            return $case->getLabel();
        }

        // Fallback to name
        return $case->name;
    }

    protected function isTranslationMissing(string $label, UnitEnum $case): bool
    {
        // If label equals the case value/name, it might be a missing translation
        // This is a heuristic - Laravel's __ function returns the key if translation is missing
        if ($case instanceof BackedEnum) {
            return $label === $case->value;
        }

        return $label === $case->name;
    }

    protected function getLocales(): array
    {
        $locales = config('sync.schema.locales');

        // If locales is a callable, invoke it
        if (is_callable($locales)) {
            return $locales();
        }

        // If locales is a class name string, resolve and call
        if (is_string($locales) && class_exists($locales)) {
            return app($locales)();
        }

        // If it's an array, return it
        if (is_array($locales)) {
            return $locales;
        }

        // Default fallback
        return [config('app.locale', 'en')];
    }

    protected function getFallbackLocale(): string
    {
        return config('sync.schema.fallback_locale', config('app.fallback_locale', 'en'));
    }
}
