<?php

declare(strict_types=1);

namespace CGA\Sync\Http\Controllers;

use CGA\Sync\Http\Requests\StoreSyncDocumentBatchRequest;
use CGA\Sync\Http\Requests\StoreSyncDocumentRequest;
use CGA\Sync\Http\Resources\SyncDocumentResource;
use CGA\Sync\Services\BatchProcessor;
use CGA\Sync\Services\DocumentProcessor;
use Illuminate\Http\JsonResponse;

class SyncDocumentController
{
    /**
     * Process a sync document
     */
    public function store(StoreSyncDocumentRequest $request, string $syncId, DocumentProcessor $processor): SyncDocumentResource
    {
        $document = $processor->process(
            syncId: $syncId,
            document: collect($request->input('data', [])),
            clientDetails: collect($request->input('client_details', [])),
            expectedRevisionId: $request->input('revision_id'),
            deleted: $request->boolean('deleted'),
        );

        return SyncDocumentResource::make($document->fresh());
    }

    /**
     * Process multiple sync documents in batch
     */
    public function batch(StoreSyncDocumentBatchRequest $request, BatchProcessor $batchProcessor): JsonResponse
    {
        $documents = collect($request->collect('documents'));
        $globalClientDetails = collect($request->input('client_details', []));

        $batchResult = $batchProcessor->process($documents, $globalClientDetails);

        return response()->json($batchResult->toArray());
    }

    /**
     * Get document status for polling
     */
    public function show(string $syncId, DocumentProcessor $processor): SyncDocumentResource
    {
        $document = $processor->findDocument($syncId);

        return SyncDocumentResource::make($document);
    }
}
