<?php

declare(strict_types=1);

namespace CGA\Sync\Services;

use CGA\Sync\Exceptions\SyncableTypeException;
use CGA\Sync\Exceptions\SyncConflictException;
use CGA\Sync\Exceptions\UnknownSyncModelException;
use CGA\Sync\Validators\ValidatorResolver;
use CGA\Sync\Values\BatchItem;
use CGA\Sync\Values\BatchResult;
use CGA\Sync\Values\ProcessedResult;
use Exception;
use Illuminate\Support\Collection;
use Illuminate\Validation\ValidationException;

class BatchProcessor
{
    public function __construct(
        private readonly DocumentProcessor $documentProcessor
    ) {}

    public function process(Collection $documents, Collection $clientDetails): BatchResult
    {
        $results = $documents->map(function ($document) use ($clientDetails) {
            $item = BatchItem::fromArray($document);

            return $this->processItem($item, $clientDetails);
        });

        return new BatchResult($results);
    }

    private function processItem(BatchItem $item, Collection $clientDetails): array
    {
        // Validate the item.
        $validationResult = $this->validateItem($item);

        if (! $validationResult['valid']) {
            return ProcessedResult::validationError(
                syncId: $item->syncId,
                errors: $validationResult['errors'],
            )->toArray();
        }

        // Process the item.
        try {
            $result = $this->documentProcessor->process(
                syncId: $item->syncId,
                document: $item->document,
                clientDetails: $clientDetails,
                expectedRevisionId: $item->revisionId,
                deleted: $item->deleted,
            );

            return ProcessedResult::success(
                syncId: $item->syncId,
                document: $result,
            )->toArray();
        } catch (SyncConflictException $e) {
            return $e->toResultArray();
        } catch (Exception $e) {
            return ProcessedResult::error(
                syncId: $item->syncId,
                message: $e->getMessage(),
            )->toArray();
        }
    }

    private function validateItem(BatchItem $item): array
    {
        try {
            ValidatorResolver::fromPayload(
                syncId: $item->syncId,
                payload: $item->document->toArray(),
                deleted: $item->deleted,
            )->validate();

            return ['valid' => true, 'errors' => []];
        } catch (ValidationException $e) {
            return ['valid' => false, 'errors' => $e->errors()];
        } catch (UnknownSyncModelException $e) {
            return ['valid' => false, 'errors' => ['syncable_type' => [$e->getMessage()]]];
        } catch (SyncableTypeException $e) {
            return ['valid' => false, 'errors' => ['syncable_type' => [$e->getMessage()]]];
        } catch (Exception $e) {
            return ['valid' => false, 'errors' => ['general' => [$e->getMessage()]]];
        }
    }
}
