<?php

declare(strict_types=1);

namespace CGA\Sync\Http\Requests;

use CGA\Sync\Validators\ValidatorResolver;
use Illuminate\Container\Attributes\RouteParameter;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\ValidationException;
use Illuminate\Validation\Validator;

class StoreSyncDocumentRequest extends FormRequest
{
    public function authorize(): bool
    {
        return true;
    }

    /**
     * @return array<string, mixed>
     */
    public function rules(): array
    {
        return [
            'data' => 'required_without:deleted|array',
            'data.syncable_type' => 'required_without:revision_id|string|max:255',
            'client_details' => 'sometimes|array',
            'revision_id' => 'sometimes|string',
            'deleted' => 'sometimes|boolean',
        ];
    }

    /**
     * @return array<string, string>
     */
    public function messages(): array
    {
        return [
            'data.required' => 'The data field is required.',
            'data.array' => 'The data must be an array.',
            'client_details.array' => 'The client details must be an array.',
            'revision_id.string' => 'The revision ID must be a string.',
            'deleted.boolean' => 'The deleted field must be a boolean.',
        ];
    }

    public function after(#[RouteParameter('syncId')] string $syncId): array
    {
        $payload = $this->input('data', []);

        return [
            function (Validator $validator) use ($syncId, $payload) {
                if ($validator->errors()->isNotEmpty()) {
                    return;
                }

                try {
                    ValidatorResolver::fromPayload(
                        syncId: $syncId,
                        payload: $payload,
                        deleted: $this->boolean('deleted'),
                    )->validate();
                } catch (ValidationException $e) {
                    // Namespace the validation errors under 'data.'
                    foreach ($e->errors() as $field => $messages) {
                        foreach ((array) $messages as $message) {
                            $validator->errors()->add("data.{$field}", $message);
                        }
                    }
                }
            },
        ];
    }
}
