<?php

declare(strict_types=1);

namespace CGA\Sync\Handlers;

use CGA\Sync\Contracts\SyncableContract;
use CGA\Sync\Enums\Operation as SyncOperation;
use CGA\Sync\Enums\Status as SyncStatus;
use CGA\Sync\Models\SyncDocument;
use Illuminate\Database\Eloquent\Model;

class DeleteHandler extends BaseHandler
{
    /**
     * Can handle documents marked for deletion.
     */
    public function canHandle(SyncDocument $syncDocument): bool
    {
        return SyncOperation::is(SyncOperation::DELETE, $syncDocument->syncable_id, $syncDocument->deleted);
    }

    /**
     * Delete the entity associated with the sync document.
     */
    protected function executeOperation(SyncDocument $syncDocument): Model
    {
        /** @var (Model&SyncableContract)|null $entity */
        $entity = $syncDocument->syncable;

        if (! $entity) {
            throw new \RuntimeException("Syncable entity not found for deletion: {$syncDocument->sync_id}");
        }

        // Generate new revision ID before deletion
        $revisionId = $this->revisionGenerator->generate($syncDocument->sync_id);

        // Delete the entity (soft delete if configured on the model)
        $entity->skipSyncEvents(fn() => $entity->delete());

        // Update sync document with new revision ID
        $syncDocument->update([
            'revision_id' => $revisionId,
            'status' => SyncStatus::APPLIED,
            'deleted' => true,
        ]);

        return $entity;
    }
}
