<?php

declare(strict_types=1);

namespace CGA\Sync\Handlers;

use CGA\Sync\Enums\Operation as SyncOperation;
use CGA\Sync\Enums\Status as SyncStatus;
use CGA\Sync\Models\SyncDocument;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\Relation;

class CreateHandler extends BaseHandler
{
    /**
     * Can handle documents that don't have an associated entity yet.
     */
    public function canHandle(SyncDocument $syncDocument): bool
    {
        return SyncOperation::is(SyncOperation::CREATE, $syncDocument->syncable_id, $syncDocument->deleted);
    }

    /**
     * Create a new entity from the sync document.
     */
    protected function executeOperation(SyncDocument $syncDocument): Model
    {
        $modelClass = $this->resolveModelClass($syncDocument->syncable_type);

        $entity = $modelClass::withoutSyncEvents(function () use ($syncDocument, $modelClass) {
            return $modelClass::create($this->prepareData(new $modelClass(), $syncDocument->document));
        });

        // Generate new revision ID
        $revisionId = $this->revisionGenerator->generateForNew($syncDocument->sync_id);

        // Update the sync document to reference the newly created entity
        $syncDocument->update([
            'syncable_id' => $entity->getKey(),
            'revision_id' => $revisionId,
            'status' => SyncStatus::APPLIED,
        ]);

        return $entity;
    }

    /**
     * Resolve the model class from the syncable_type.
     */
    protected function resolveModelClass(string $syncableType): string
    {
        // Handle both morph map keys and full class names
        $modelClass = Relation::getMorphedModel($syncableType) ?? $syncableType;

        if (! class_exists($modelClass)) {
            throw new \InvalidArgumentException("Model class does not exist: {$modelClass}");
        }

        return $modelClass;
    }
}
