<?php

declare(strict_types=1);

namespace CGA\Sync\Values;

use CGA\Sync\Contracts\SyncableContract;
use CGA\Sync\Exceptions\UnknownSyncModelException;
use CGA\Sync\Services\ModelStructureService;
use Illuminate\Container\Attributes\RouteParameter;
use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\Relation;
use Illuminate\Support\Collection;

class SyncableEntityType implements Arrayable
{
    public function __construct(#[RouteParameter('type')] public private(set) string $type)
    {
        $class = $this->getMorphedModel();

        if (! $class) {
            throw UnknownSyncModelException::forType($type);
        }

        if (! is_subclass_of($class, SyncableContract::class)) {
            throw UnknownSyncModelException::forType($type);
        }
    }

    public function getMorphedModel(): ?string
    {
        return once(fn() => Relation::getMorphedModel($this->type));
    }

    public function query(): Builder
    {
        $query = $this->getMorphedModel()::query()
            ->with([
                'syncDocument',
            ]);

        // I need to use the service to get the relations and eager load them automatically.
        $modelStructure = app(ModelStructureService::class)->getStructure(
            $this->getMorphedModel()
        );

        // Get the relations and eager load them automatically.
        $relations = collect($modelStructure['relations'])
            ->pluck('name')
            ->map(fn($relation) => [
                $relation,
                "{$relation}.syncDocument",
            ]);

        foreach ($relations as $relation) {
            $query->with($relation);
        }

        return $query;
    }

    public function toArray(): array
    {
        return [
            'type' => $this->type,
        ];
    }

    public static function listTypes(): Collection
    {
        return once(
            fn() => collect(Relation::morphMap())
                ->filter(function ($class) {
                    return is_subclass_of($class, SyncableContract::class);
                })
                ->keys()
        );
    }

    public static function list(): Collection
    {
        return self::listTypes()->map(fn($type) => new self($type));
    }
}
