<?php

declare(strict_types=1);

namespace CGA\Sync\Values;

use CGA\Sync\Enums\SyncResultStatus;
use CGA\Sync\Models\SyncDocument;
use Illuminate\Contracts\Support\Arrayable;

class ProcessedResult implements Arrayable
{
    private function __construct(
        public readonly string $syncId,
        public readonly SyncResultStatus $status,
        public readonly string $message,
        public readonly ?string $currentRevision = null,
        public readonly ?array $currentState = null,
        public readonly ?array $validationErrors = null,
    ) {}

    public static function success(string $syncId, SyncDocument $document): self
    {
        return new self(
            syncId: $syncId,
            status: SyncResultStatus::SUCCESS,
            message: __('Processing successful'),
            currentRevision: $document->revision_id,
            currentState: $document->document->toArray(),
        );
    }

    public static function validationError(string $syncId, array $errors): self
    {
        return new self(
            syncId: $syncId,
            status: SyncResultStatus::VALIDATION_ERROR,
            message: __('Validation failed'),
            validationErrors: $errors,
        );
    }

    public static function error(string $syncId, string $message): self
    {
        return new self(
            syncId: $syncId,
            status: SyncResultStatus::ERROR,
            message: $message,
        );
    }

    public static function conflict(string $syncId, string $message, ?string $currentRevision = null, ?array $currentState = null): self
    {
        return new self(
            syncId: $syncId,
            status: SyncResultStatus::CONFLICT,
            message: $message,
            currentRevision: $currentRevision,
            currentState: $currentState,
        );
    }

    public function toArray(): array
    {
        $result = [
            'sync_id' => $this->syncId,
            'status' => $this->status->value,
            'message' => $this->message,
            'current_revision' => $this->currentRevision,
            'current_state' => $this->currentState,
        ];

        if ($this->validationErrors !== null) {
            $result['validation_errors'] = $this->validationErrors;
        }

        return $result;
    }
}
