<?php

declare(strict_types=1);

namespace CGA\Sync\Validators;

use CGA\Sync\Exceptions\SyncableTypeException;
use CGA\Sync\Facades\Sync;
use CGA\Sync\Models\SyncDocument;
use CGA\Sync\Values\SyncableEntityType;
use Illuminate\Support\Arr;

class ValidatorResolver
{
    public function resolve(SyncDocument $syncDocument, array $payload, ?bool $deleted = false): AbstractValidator
    {
        $modelClass = new SyncableEntityType($syncDocument->syncable_type)->getMorphedModel();
        $validatorClass = $this->getValidatorClass($modelClass);

        return app($validatorClass, [
            'syncDocument' => $syncDocument,
            'payload' => $payload,
            'deleted' => $deleted,
        ]);
    }

    protected function getValidatorClass(string $modelClass): string
    {
        $modelName = class_basename($modelClass);

        foreach ($this->getValidatorNamespaces() as $namespace) {
            $validatorClass = "{$namespace}{$modelName}Validator";

            if (class_exists($validatorClass)) {
                return $validatorClass;
            }
        }

        return BaseValidator::class;
    }

    protected function getValidatorNamespaces(): array
    {
        $defaultNamespaces = [
            'App\\Support\\Sync\\Validators\\',
        ];

        return config('sync.validator_namespaces', $defaultNamespaces) ?? $defaultNamespaces;
    }

    public static function getSyncableTypeFrom(array $payload): string
    {
        $syncableType = Arr::get($payload, 'syncable_type');

        if (! $syncableType) {
            throw SyncableTypeException::missing();
        }

        return $syncableType;
    }

    public static function fromPayload(string $syncId, ?array $payload = [], ?bool $deleted = false): AbstractValidator
    {
        $documentClass = Sync::getDocumentModel();
        $syncDocument = $documentClass::firstWhere('sync_id', $syncId);

        if (! $syncDocument) {
            $syncableType = self::getSyncableTypeFrom($payload);

            $syncDocument = new $documentClass([
                'sync_id' => $syncId,
                'syncable_type' => $syncableType,
                'syncable_id' => Arr::get($payload, 'syncable_id'),
                'document' => $payload,
                'deleted' => $deleted,
            ]);
        }

        return new self()->resolve(
            syncDocument: $syncDocument,
            payload: $payload,
            deleted: $deleted,
        );
    }
}
