<?php

declare(strict_types=1);

namespace CGA\Sync\Services;

use CGA\Sync\Facades\Sync;
use Ramsey\Uuid\Uuid;

class RevisionIdGenerator
{
    /**
     * Generate a new revision ID for the given sync_id.
     * Format: [count-xxh3(sync_id)]
     */
    public function generate(string|Uuid $syncId): string
    {
        // Get current revision count for this sync_id
        $currentCount = $this->getCurrentRevisionCount($syncId);
        $newCount = $currentCount + 1;

        // Generate deterministic hash from sync_id
        $hash = $this->generateHash($syncId);

        return "{$newCount}-{$hash}";
    }

    /**
     * Generate a revision ID for a new sync document (count = 1).
     */
    public function generateForNew(string|Uuid $syncId): string
    {
        $hash = $this->generateHash($syncId);

        return "1-{$hash}";
    }

    /**
     * Get the current revision count from existing sync documents.
     */
    protected function getCurrentRevisionCount(string|Uuid $syncId): int
    {
        $currentRevision = Sync::getDocumentModel()::where('sync_id', $syncId)
            ->max('revision_id');

        if (! $currentRevision) {
            return 0;
        }

        // Extract count from revision format "3-abc12345"
        $parts = explode('-', $currentRevision);

        return (int) $parts[0];
    }

    /**
     * Generate xxh3 hash (first 8 characters).
     */
    protected function generateHash(string|Uuid $input): string
    {
        return substr(hash('xxh3', (string) $input), 0, 8);
    }
}
