<?php

declare(strict_types=1);

namespace CGA\Sync\Listeners;

use CGA\Sync\Contracts\DocumentHandler;
use CGA\Sync\Enums\Status as SyncStatus;
use CGA\Sync\Events\SyncDocumentSaved;
use CGA\Sync\Handlers\CreateHandler;
use CGA\Sync\Handlers\DeleteHandler;
use CGA\Sync\Handlers\UpdateHandler;
use CGA\Sync\Services\RevisionIdGenerator;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Log;

class ProcessSyncDocumentListener implements ShouldQueue
{
    protected Collection $handlers;

    public function __construct(RevisionIdGenerator $revisionGenerator)
    {
        $this->handlers = collect([
            new CreateHandler($revisionGenerator),
            new UpdateHandler($revisionGenerator),
            new DeleteHandler($revisionGenerator),
        ]);
    }

    /**
     * Handle the event.
     */
    public function handle(SyncDocumentSaved $event): void
    {
        $syncDocument = $event->syncDocument;

        // Skip processing if already applied
        if ($syncDocument->status === SyncStatus::APPLIED) {
            return;
        }

        try {
            // Find the appropriate handler for this sync document
            $handler = $this->handlers->first(
                fn(DocumentHandler $handler) => $handler->canHandle($syncDocument)
            );

            if (! $handler) {
                throw new \RuntimeException("No handler found for sync document: {$syncDocument->sync_id}");
            }

            // Process the document using the handler (includes revision management and status update)
            $handler->handle($syncDocument);
        } catch (\Exception $e) {
            Log::error('SyncDocument processing failed via listener', [
                'sync_id' => $syncDocument->sync_id,
                'syncable_type' => $syncDocument->syncable_type,
                'error' => $e->getMessage(),
            ]);

            throw $e;
        }
    }

    public function shouldQueue(SyncDocumentSaved $event): bool
    {
        $shouldQueue = Config::get('sync.async') === true;

        if ($shouldQueue) {
            return true;
        }

        // If async is disabled, handle the document immediately
        $this->handle($event);

        return false;
    }
}
