<?php

declare(strict_types=1);

namespace CGA\Sync\Http\Controllers;

use CGA\Sync\Collections\StateCollection;
use CGA\Sync\Http\Resources\SyncableEntityResource;
use CGA\Sync\Http\Resources\SyncableEntityTypeResource;
use CGA\Sync\Values\SyncableEntityType;
use Illuminate\Http\Request;

class StateController
{
    public function index(Request $request)
    {
        $syncableEntityTypes = SyncableEntityType::list();

        return SyncableEntityTypeResource::collection($syncableEntityTypes)->additional([
            'meta' => [
                'server_timestamp' => now()->toIso8601ZuluString(),
            ],
        ]);
    }

    public function show(Request $request, SyncableEntityType $type)
    {
        // TODO: Review if this is how we want to handle this.
        $request->user()?->can('view', $type->getMorphedModel());

        $request->validate([
            'limit' => [
                'integer',
                'min:1',
                'max:1000',
            ],
        ]);

        $entities = $type
            ->query()
            // TODO: We need to be able to inject a custom query.
            // Probably a class which we inject here based on the type.
            // Ensuring users only have access to the data they should.
            ->orderBy('id', 'desc')
            ->cursorPaginate(
                perPage: $request->input('limit', 100)
            );

        $stateCollection = StateCollection::fromPaginator($entities);
        $transformedData = $stateCollection->transformToStateData();

        return SyncableEntityResource::collection(
            resource: $entities->setCollection($transformedData) // Replace paginator with transformed data
        )->additional([
            'meta' => [
                'has_more' => $entities->hasMorePages(),
                'entity_type' => $type->type,
                'server_timestamp' => now()->toIso8601ZuluString(),
            ],
        ]);
    }
}
