<?php

declare(strict_types=1);

namespace CGA\Sync\Handlers;

use CGA\Sync\Contracts\DocumentHandler;
use CGA\Sync\Contracts\SyncableContract;
use CGA\Sync\Data\StateResolver;
use CGA\Sync\Enums\Status;
use CGA\Sync\Facades\Sync;
use CGA\Sync\Models\SyncDocument;
use CGA\Sync\Models\SyncRevision;
use CGA\Sync\Services\RevisionIdGenerator;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;

abstract class BaseHandler implements DocumentHandler
{
    public function __construct(
        protected RevisionIdGenerator $revisionGenerator
    ) {}

    /**
     * Execute handler operation with revision management.
     */
    final public function handle(SyncDocument $syncDocument): Model
    {
        try {
            return DB::transaction(function () use ($syncDocument) {
                // Perform the handler-specific operation
                $entity = $this->executeOperation($syncDocument);

                // Save successful revision to sync_revisions
                $this->saveSuccessfulRevision($syncDocument);

                return $entity;
            });
        } catch (\Exception $e) {
            // Handle failure OUTSIDE the transaction so changes persist
            $this->handleFailure($syncDocument, $e);
            throw $e;
        }
    }

    /**
     * Handler-specific operation implementation.
     */
    abstract protected function executeOperation(SyncDocument $syncDocument): Model;

    /**
     * Save successful operation to sync_revisions table.
     */
    protected function saveSuccessfulRevision(SyncDocument $syncDocument): void
    {
        Sync::saveRevision($syncDocument);
    }

    /**
     * Handle operation failure - revert UPDATE operations to last good state.
     * @throws \Exception
     */
    protected function handleFailure(SyncDocument $syncDocument, \Exception $e): void
    {
        // For UPDATE operations, try to revert to last successful state
        if ($syncDocument->syncable_id !== null) {
            /** @var SyncRevision|null $latestRevision */
            $latestRevision = $syncDocument->revisions()->latest()->first();

            if ($latestRevision) {
                // Revert to last known good state from sync_revisions
                $syncDocument->updateQuietly([
                    'status' => Status::APPLIED,
                    'revision_id' => $latestRevision->revision_id,
                    'document' => $latestRevision->document,
                    'error_message' => $e->getMessage(),
                ]);
            } else {
                // No previous revisions found, just log error
                $syncDocument->updateQuietly(['error_message' => $e->getMessage()]);
            }
        } else {
            // For CREATE operations, just log error (leave as PENDING for retry)
            $syncDocument->updateQuietly(['error_message' => $e->getMessage()]);
        }

        throw $e;
    }

    protected function prepareData(Model & SyncableContract $entity, Collection $data): array
    {
        $stateResolver = app(StateResolver::class);
        $state = $stateResolver->resolve($entity);
        $syncData = $data->except(['syncable_type'])->toArray();

        return $state->prepareForAssignment($syncData);
    }
}
