<?php

declare(strict_types=1);

namespace CGA\Sync\Data;

use CGA\Sync\Contracts\SyncableContract;
use CGA\Sync\Enums\Status as SyncStatus;
use CGA\Sync\Services\ModelStructureService;
use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Database\Eloquent\Model;

class BaseState implements Arrayable
{
    public private(set) string $id;

    protected ModelStructureService $modelStructureService;

    public function __construct(
        protected Model & SyncableContract $model
    ) {
        $this->id = $model->getSyncId();
        $this->modelStructureService = app(ModelStructureService::class);
    }

    public function getFieldData(): array
    {
        $modelStructure = $this->getModelStructure();
        $data = [];

        // process regular attributes
        foreach ($modelStructure['attributes'] as $attribute) {
            $data[$attribute['name']] = $this->resolveFieldValue($attribute);
        }

        // process relations
        foreach ($modelStructure['relations'] as $relation) {
            $data[$relation['name']] = $this->resolveRelationFieldValue($relation);
        }

        return $data;
    }

    public function getRevisionId(): ?string
    {
        $syncDocument = $this->model->getSyncDocument();

        return $syncDocument && $syncDocument->status === SyncStatus::APPLIED
            ? $syncDocument->revision_id
            : null;
    }

    public function getModelStructure(): array
    {
        return $this->modelStructureService->getStructure(
            get_class($this->model),
            $this->stateExcludes()
        );
    }

    public function prepareForAssignment(array $syncData): array
    {
        $modelStructure = $this->getModelStructure();
        $preparedData = $syncData;

        foreach ($preparedData as $key => $value) {
            if ($this->shouldExclude($key, $this->excludes())) {
                unset($preparedData[$key]);
            }
        }

        // Process BelongsTo relationships: convert sync IDs to foreign keys
        foreach ($modelStructure['relations'] as $relation) {
            if ($relation['type'] !== 'BelongsTo') {
                continue;
            }

            $relationName = $relation['name'];

            if (! array_key_exists($relationName, $syncData)) {
                continue;
            }

            $syncId = $syncData[$relationName];
            $foreignKey = $this->model->{$relationName}()->getForeignKeyName();

            if ($syncId === null) {
                $preparedData[$foreignKey] = null;
            } else {
                // Find referenced entity - must exist due to creation order requirement
                $relatedModel = $relation['related'];
                $relatedEntity = $relatedModel::bySyncId($syncId)->first();

                if (! $relatedEntity) {
                    throw new \InvalidArgumentException("Referenced entity with sync_id {$syncId} not found for relation {$relationName}");
                }

                $preparedData[$foreignKey] = $relatedEntity->id;
            }

            unset($preparedData[$relationName]);
        }

        return $preparedData;
    }

    protected function shouldExclude(string $fieldName, ?array $excludes = null): bool
    {
        return $this->modelStructureService->shouldExclude($fieldName, $excludes ?? $this->excludes());
    }

    protected function resolveFieldValue(array $attribute): mixed
    {
        $fieldName = $attribute['name'];

        return $this->model->{$fieldName};
    }

    protected function resolveRelationFieldValue(array $attribute): mixed
    {
        $relationName = $attribute['name'];

        $relation = $this->model->{$relationName};

        if (is_null($relation)) {
            return null;
        }

        return $relation->getSyncId();
    }

    protected function excludes(): array
    {
        return [
            'id',
            'created_at',
            'updated_at',
            'deleted_at',
            'password',
            'remember_token',
            'email_verified_at',
            '/two_factor_/', // Regex for two_factor_* fields
        ];
    }

    protected function stateExcludes(): array
    {
        return [
            ...$this->excludes(),
            '/.*_id$/',  // Regex for fields ending with _id
        ];
    }

    public function toArray(): array
    {
        return [
            'sync_id' => $this->id,
            'data' => $this->getFieldData(),
            'revision_id' => $this->getRevisionId(),
        ];
    }
}
