<?php

declare(strict_types=1);

namespace CGA\Sync;

use Illuminate\Auth\AuthManager;
use Illuminate\Contracts\Auth\Authenticatable;
use Illuminate\Contracts\Auth\StatefulGuard;
use Illuminate\Contracts\Config\Repository;

class CauserResolver
{
    protected ?string $authDriver;

    public function __construct(Repository $config, protected AuthManager $authManager)
    {
        $this->authDriver = $config->get('sync.default_auth_driver');
    }

    public function resolve(Authenticatable | int | string | null $subject = null): ?Authenticatable
    {
        return $this->getCauser($subject);
    }

    protected function resolveUsingId(int | string $subject): ?Authenticatable
    {
        $guard = $this->authManager->guard($this->authDriver);

        if (! $guard instanceof StatefulGuard) {
            return null;
        }

        $provider = method_exists($guard, 'getProvider') ? $guard->getProvider() : null;
        $model = method_exists($provider, 'retrieveById') ? $provider->retrieveById($subject) : null;

        return $model;
    }

    protected function getCauser(Authenticatable | int | string | null $subject = null): ?Authenticatable
    {
        if ($subject instanceof Authenticatable) {
            return $subject;
        }

        if (is_null($subject)) {
            return $this->getDefaultCauser();
        }

        return $this->resolveUsingId($subject);
    }

    protected function getDefaultCauser(): ?Authenticatable
    {
        return $this->authManager->guard($this->authDriver)->user();
    }
}
