<?php

declare(strict_types=1);

namespace CGA\Sync\Validators\Rules;

use CGA\Sync\Enums\Operation as SyncOperation;

class Generator
{
    public function __construct(
        public private(set) array $modelStructure,
        public private(set) SyncOperation $operation,
    ) {}

    /**
       * Generate basic CREATE rules from model structure
       */
    public function generateCreateRules(): array
    {
        $rules = [];

        foreach ($this->modelStructure['attributes'] as $attribute) {
            $fieldRules = $this->generateCreateAttributeRules($attribute);

            if (! empty($fieldRules)) {
                $rules[$attribute['name']] = $fieldRules;
            }
        }

        return $rules;
    }

    /**
     * Generate basic UPDATE rules from model structure
     */
    public function generateUpdateRules(): array
    {
        $rules = [];

        foreach ($this->modelStructure['attributes'] as $attribute) {
            $fieldRules = $this->generateUpdateAttributeRules($attribute);

            if (! empty($fieldRules)) {
                $rules[$attribute['name']] = $fieldRules;
            }
        }

        return $rules;
    }

    protected function generateCreateAttributeRules(array $attribute): array
    {
        $rules = [];
        $nullable = $attribute['nullable'];

        // Handle required/nullable for CREATE
        if ($nullable) {
            $rules[] = 'nullable';
        } else {
            $rules[] = 'required';
        }

        // Add type-specific rules
        $typeRules = $this->getTypeRules(
            $attribute['type'],
            $attribute['cast'] ?? null
        );
        $rules = array_merge($rules, $typeRules);

        return $rules;
    }

    protected function generateUpdateAttributeRules(array $attribute): array
    {
        // If the attribute is a computed attribute, we don't need to validate it
        if (array_key_exists('cast', $attribute) && $attribute['cast'] === 'attribute') {
            return [];
        }

        // Also check if type is null (no database column)
        if ($attribute['type'] === null) {
            return [];
        }

        $rules = [];
        $nullable = $attribute['nullable'];
        $isArray = in_array($attribute['cast'] ?? '', ['array', 'json', 'collection']);

        // Handle UPDATE: fields are optional but validate if present
        $rules[] = 'sometimes';

        if ($nullable) {
            $rules[] = 'nullable';
        } else {
            // For arrays, use 'present' instead of 'required' (allows empty arrays)
            if ($isArray) {
                $rules[] = 'present';
            } else {
                $rules[] = 'required';
            }
        }

        // Add type-specific rules
        $typeRules = $this->getTypeRules(
            $attribute['type'],
            $attribute['cast'] ?? null
        );
        $rules = array_merge($rules, $typeRules);

        return $rules;
    }

    protected function getTypeRules(string $type, ?string $cast = null): array
    {
        // If there's a cast defined, use that instead of the database type
        if ($cast) {
            return $this->getRulesForCastType($cast);
        }

        // Fall back to database type rules
        $baseType = $this->extractBaseType($type);
        $length = $this->extractLength($type);

        return match ($baseType) {
            // String types
            'varchar', 'char' => [
                'string',
                $length ? "max:{$length}" : 'max:255',
            ],

            'string', 'text', 'longtext', 'mediumtext' => [
                'string',
            ],

            // Integer types
            'int', 'integer', 'bigint', 'smallint', 'tinyint', 'mediumint' => [
                'integer',
            ],

            // Numeric types
            'decimal', 'numeric' => [
                'numeric',
            ],

            'float', 'double', 'real' => [
                'numeric',
            ],

            // Boolean types
            'boolean', 'bool', 'tinyint(1)' => [
                'boolean',
            ],

            // JSON types
            'json', 'jsonb' => [
                'array', // Laravel converts JSON to arrays in requests
            ],

            // Date types
            'date' => [
                'date',
            ],

            'datetime', 'timestamp' => [
                'date',
            ],

            'time' => [
                'date_format:H:i:s',
            ],

            'year' => [
                'integer', 'between:1901,2155', // MySQL YEAR range
            ],

            // Binary types - use string as fallback for form validation
            'binary', 'varbinary', 'blob', 'mediumblob', 'longblob' => [
                'string',
            ],

            // UUID/ULID types
            'uuid' => [
                'string', 'uuid',
            ],

            'ulid' => [
                'string', 'ulid',
            ],

            // Enum/Set types - validate as string since values are pre-defined
            'enum', 'set' => [
                'string',
            ],

            // Geometry types - validate as string (WKT format typically)
            'geometry', 'point', 'linestring', 'polygon', 'multipoint',
            'multilinestring', 'multipolygon', 'geometrycollection' => [
                'string',
            ],

            default => [
                'string', // Safe fallback
            ]
        };
    }

    protected function getRulesForCastType(string $castType): array
    {
        return match ($castType) {
            'array', 'json', 'collection' => ['array'],
            'object' => ['array'], // Objects come as arrays from requests
            'boolean', 'bool' => ['boolean'],
            'integer', 'int' => ['integer'],
            'float', 'double', 'real' => ['numeric'],
            'decimal' => ['numeric'],
            'date' => ['date'],
            'datetime' => ['date'],
            'timestamp' => ['date'],
            default => ['string'], // Fallback for custom casts
        };
    }

    protected function extractBaseType(string $type): string
    {
        // Handle types like "varchar(255)", "decimal(10,2)"
        if (preg_match('/^([a-zA-Z]+)/', $type, $matches)) {
            return strtolower($matches[1]);
        }

        return strtolower($type);
    }

    protected function extractLength(string $type): ?int
    {
        // Extract length from types like "varchar(255)"
        if (preg_match('/\((\d+)(?:,\d+)?\)/', $type, $matches)) {
            return (int) $matches[1];
        }

        return null;
    }

}
