<?php

declare(strict_types=1);

namespace CGA\Sync\Validators;

use CGA\Sync\Data\StateResolver;
use CGA\Sync\Enums\Operation as SyncOperation;
use CGA\Sync\Models\SyncDocument;
use CGA\Sync\Values\SyncableEntityType;
use Illuminate\Support\Facades\Validator;

abstract class AbstractValidator
{
    public private(set) SyncOperation $operation;

    public function __construct(
        protected SyncDocument $syncDocument,
        protected array $payload,
        protected ?bool $deleted = false,
    ) {
        $this->operation = SyncOperation::infer(
            syncableId: $this->syncDocument->syncable_id,
            deleted: $this->deleted,
        );
    }

    abstract protected function createRules(): array;

    abstract protected function updateRules(): array;

    abstract protected function authorize(): void;

    abstract protected function validateBusinessRules(): void;

    protected function deleteRules(): array
    {
        return [];
    }

    public function validate(): void
    {
        $this->authorize();
        $this->validateSchema();
        $this->validateBusinessRules();
    }

    protected function validateSchema(): void
    {
        $rules = match ($this->operation) {
            SyncOperation::CREATE => $this->createRules(),
            SyncOperation::UPDATE => $this->updateRules(),
            SyncOperation::DELETE => $this->deleteRules(),
        };

        if (empty($rules)) {
            return;
        }

        $validator = Validator::make(
            $this->payload,
            $rules
        );

        $validator->validate();
    }

    protected function getModelStructure(): array
    {
        // TODO: Review if we can put this in a once() block.
        $syncableType = $this->syncDocument->syncable_type;
        $entityClass = new SyncableEntityType($syncableType)->getMorphedModel();
        $entity = new $entityClass();

        return app(StateResolver::class)->resolve($entity)->getModelStructure();
    }

    protected function getGenerator(): Rules\Generator
    {
        return new Rules\Generator(
            modelStructure: $this->getModelStructure(),
            operation: $this->operation
        );
    }
}
