<?php

declare(strict_types=1);

namespace CGA\Sync;

use CGA\Sync\Events\SyncableModelCreated;
use CGA\Sync\Events\SyncableModelDeleted;
use CGA\Sync\Events\SyncableModelRestored;
use CGA\Sync\Events\SyncableModelUpdated;
use CGA\Sync\Events\SyncDocumentCreating;
use CGA\Sync\Events\SyncDocumentSaved;
use CGA\Sync\Listeners\PrepareSyncDocument;
use CGA\Sync\Listeners\ProcessServerSyncDocumentListener;
use CGA\Sync\Listeners\ProcessSyncDocumentListener;
use Illuminate\Contracts\Foundation\CachesRoutes;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\Facades\Route;
use Spatie\LaravelPackageTools\Package;
use Spatie\LaravelPackageTools\PackageServiceProvider;

class SyncServiceProvider extends PackageServiceProvider
{
    public function configurePackage(Package $package): void
    {
        /*
         * This class is a Package Service Provider
         *
         * More info: https://github.com/spatie/laravel-package-tools
         */
        $package
            ->name('sync')
            ->hasConfigFile()
            ->hasMigrations('create_sync_documents_table', 'create_sync_revisions_table');
    }

    public function packageBooted(): void
    {
        $this->listenToEvents();
        $this->registerRoutes();
    }

    public function registerPackage()
    {
        $this->app->scoped(CauserResolver::class);
    }

    private function listenToEvents(): void
    {
        Event::listen(
            [
                SyncableModelCreated::class,
                SyncableModelUpdated::class,
                SyncableModelDeleted::class,
                SyncableModelRestored::class,
            ],
            ProcessServerSyncDocumentListener::class,
        );

        Event::listen(
            SyncDocumentSaved::class,
            ProcessSyncDocumentListener::class,
        );

        Event::listen(
            SyncDocumentCreating::class,
            PrepareSyncDocument::class,
        );
    }

    private function registerRoutes(): void
    {
        if ($this->app instanceof CachesRoutes && $this->app->routesAreCached()) {
            return;
        }

        Route::group([
            'prefix' => config('sync.route.prefix'),
            'middleware' => config('sync.route.middleware', ['api']),
            'as' => 'sync.',
        ], function () {
            $this->loadRoutesFrom(__DIR__ . '/../routes/api.php');
        });
    }
}
