<?php

declare(strict_types=1);

namespace CGA\Sync\Services;

use CGA\Sync\Contracts\SyncableContract;
use Illuminate\Database\Eloquent\ModelInspector;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class ModelStructureService
{
    private static array $cache = [];

    public function __construct(
        private ModelInspector $inspector
    ) {}

    public function getStructure(string $modelClass, array $excludes = []): array
    {
        $cacheKey = $modelClass . '|' . hash('xxh3', serialize($excludes));

        if (isset(self::$cache[$cacheKey])) {
            return self::$cache[$cacheKey];
        }

        $structure = [
            'attributes' => [],
            'relations' => [],
        ];

        $inspectResult = $this->inspector->inspect($modelClass);

        // Get all attributes
        foreach ($inspectResult['attributes'] as $info) {
            $name = $info['name'];

            if ($this->shouldExclude($name, $excludes)) {
                continue;
            }

            $structure['attributes'][] = $info;
        }

        // Get BelongsTo relations only (they have foreign keys that need UUID conversion)
        foreach ($inspectResult['relations'] as $info) {
            if (! is_subclass_of($info['related'], SyncableContract::class)) {
                continue;
            }

            if ($info['type'] !== class_basename(BelongsTo::class)) {
                continue;
            }

            $structure['relations'][] = $info;
        }

        self::$cache[$cacheKey] = $structure;

        return $structure;
    }

    public function shouldExclude(string $fieldName, array $excludes): bool
    {
        foreach ($excludes as $exclude) {
            if ($fieldName === $exclude) {
                return true;
            }

            if (str_starts_with($exclude, '/') && str_ends_with($exclude, '/')) {
                if (preg_match($exclude, $fieldName)) {
                    return true;
                }
            }
        }

        return false;
    }
}
