<?php

declare(strict_types=1);

namespace CGA\Sync\Handlers;

use CGA\Sync\Contracts\SyncableContract;
use CGA\Sync\Enums\Operation as SyncOperation;
use CGA\Sync\Enums\Status as SyncStatus;
use CGA\Sync\Models\SyncDocument;
use Illuminate\Database\Eloquent\Model;

class UpdateHandler extends BaseHandler
{
    /**
     * Can handle documents that have an associated entity and are not deleted.
     */
    public function canHandle(SyncDocument $syncDocument): bool
    {
        return SyncOperation::is(SyncOperation::UPDATE, $syncDocument->syncable_id, $syncDocument->deleted);
    }

    /**
     * Update the existing entity with data from the sync document.
     */
    protected function executeOperation(SyncDocument $syncDocument): Model
    {
        /** @var (Model&SyncableContract)|null $entity */
        $entity = $syncDocument->syncable;

        if (! $entity) {
            throw new \RuntimeException("Syncable entity not found for sync document: {$syncDocument->sync_id}");
        }

        // Update the entity using wholesale assignment - Laravel casting handles everything
        $entity->skipSyncEvents(fn() => $entity->update($this->prepareData($entity, $syncDocument->document)));

        // Generate new revision ID
        $revisionId = $this->revisionGenerator->generate($syncDocument->sync_id);

        // Update sync document with new revision ID
        $syncDocument->update([
            'revision_id' => $revisionId,
            'status' => SyncStatus::APPLIED,
        ]);

        return $entity;
    }
}
