<?php

declare(strict_types=1);

namespace CGA\Sync\Models;

use Carbon\CarbonInterface;
use CGA\Sync\Enums\Status;
use CGA\Sync\Events\SyncDocumentCreating;
use CGA\Sync\Events\SyncDocumentSaved;
use CGA\Sync\Facades\Sync;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Support\Collection;

/**
 * @property string $sync_id
 * @property string $revision_id
 * @property Collection $document
 * @property Collection $client_details
 * @property string $syncable_type
 * @property int|null $syncable_id
 * @property Status $status
 * @property bool $deleted
 * @property CarbonInterface $updated_at
 * @property CarbonInterface $created_at
 * @property string|null $causer_type
 * @property int|null $causer_id
 */
class SyncDocument extends Model
{
    /** @use HasFactory<\CGA\Sync\Database\Factories\SyncDocumentFactory> */
    use HasFactory;

    protected $guarded = [];

    protected $attributes = [
        'document' => '{}',
        'client_details' => '{}',
        'deleted' => false,
        'status' => Status::PENDING,
    ];

    protected $dispatchesEvents = [
        'saved' => SyncDocumentSaved::class,
        'creating' => SyncDocumentCreating::class,
    ];

    protected function casts(): array
    {
        return [
            'document' => 'collection',
            'client_details' => 'collection',
            'status' => Status::class,
            'deleted' => 'boolean',
        ];
    }

    public function __construct(array $attributes = [])
    {
        if (! isset($this->table)) {
            $this->setTable(config('sync.table_names.sync_documents'));
        }

        parent::__construct($attributes);
    }

    /** @return MorphTo<Model, $this> */
    public function syncable(): MorphTo
    {
        return $this->morphTo();
    }

    /** @return HasMany<SyncRevision, $this> */
    public function revisions(): HasMany
    {
        return $this->hasMany(Sync::getRevisionModel(), 'sync_id', 'sync_id');
    }

    /** @return MorphTo<Model, $this> */
    public function causer(): MorphTo
    {
        return $this->morphTo();
    }
}
