<?php

declare(strict_types=1);

namespace CGA\Sync\Listeners;

use CGA\Sync\Data\StateResolver;
use CGA\Sync\Enums\Status;
use CGA\Sync\Events\SyncableModelCreated;
use CGA\Sync\Events\SyncableModelDeleted;
use CGA\Sync\Events\SyncableModelRestored;
use CGA\Sync\Events\SyncableModelUpdated;
use CGA\Sync\Facades\Sync;
use CGA\Sync\Services\RevisionIdGenerator;

class ProcessServerSyncDocumentListener
{
    public function __construct(
        protected StateResolver $stateResolver,
        protected RevisionIdGenerator $revisionGenerator
    ) {}

    /**
     * Handle syncable model events
     * All events are server-side changes that need sync processing
     */
    public function handle(SyncableModelCreated|SyncableModelUpdated|SyncableModelDeleted|SyncableModelRestored $event): void
    {
        // Fail fast - don't catch exceptions, let them bubble up to fail the original operation
        match (true) {
            $event instanceof SyncableModelCreated => $this->handleCreate($event),
            $event instanceof SyncableModelUpdated => $this->handleUpdate($event),
            $event instanceof SyncableModelDeleted => $this->handleDelete($event),
            $event instanceof SyncableModelRestored => $this->handleRestore($event),
        };
    }

    /**
     * Handle model creation
     */
    protected function handleCreate(SyncableModelCreated $event): void
    {
        $this->upsertSyncDocument($event->model, 'create', deleted: false);
    }

    /**
     * Handle model update
     */
    protected function handleUpdate(SyncableModelUpdated $event): void
    {
        $this->upsertSyncDocument($event->model, 'update', deleted: false);
    }

    /**
     * Handle model deletion
     */
    protected function handleDelete(SyncableModelDeleted $event): void
    {
        $this->upsertSyncDocument($event->model, 'delete', deleted: true);
    }

    /**
     * Handle model restoration
     */
    protected function handleRestore(SyncableModelRestored $event): void
    {
        $this->upsertSyncDocument($event->model, 'restore', deleted: false);
    }

    /**
     * Upsert a SyncDocument to represent the server-side change
     */
    protected function upsertSyncDocument(
        $model,
        string $operation,
        bool $deleted = false
    ): void {
        // Get sync_id for the model
        $syncId = $model->getSyncId();

        // Get current state using StateResolver (complete, clean data)
        $currentState = $this->stateResolver->resolve($model);

        // Get current document (if exists) to generate proper revision
        $existingDocument = Sync::getDocumentModel()::where('sync_id', $syncId)->first();

        // Generate next revision ID
        $revisionId = $existingDocument
            ? $this->revisionGenerator->generate($syncId)
            : $this->revisionGenerator->generateForNew($syncId);

        // Prepare document data (use current state for complete data)
        $documentData = $deleted ? [] : $currentState->getFieldData();

        $syncDocument = Sync::upsertSyncDocument($syncId, [
            'syncable_type' => $model->getMorphClass(),
            'syncable_id' => $model->getKey(),
            'revision_id' => $revisionId,
            'document' => $documentData,
            'deleted' => $deleted,
            'status' => Status::APPLIED, // Server changes are immediately applied
            'client_details' => $this->getServerSideClientDetails($operation),
        ]);

        Sync::saveRevision($syncDocument);
    }

    private function getServerSideClientDetails(string $operation): array
    {
        return [
            'source' => 'server',
            'operation' => $operation,
            'timestamp' => now()->toIso8601ZuluString(),
            'user_agent' => request()->userAgent() ?? 'server-side',
        ];
    }
}
