<?php

declare(strict_types=1);

namespace CGA\Sync\Http\Controllers;

use CGA\Sync\Collections\StateCollection;
use CGA\Sync\Http\Resources\SyncableEntityResource;
use CGA\Sync\Http\Resources\SyncableEntityTypeResource;
use CGA\Sync\Values\SyncableEntityType;
use Illuminate\Http\Request;

class StateController
{
    public function index(Request $request)
    {
        $syncableEntityTypes = SyncableEntityType::list();

        return SyncableEntityTypeResource::collection($syncableEntityTypes)->additional([
            'meta' => [
                'server_timestamp' => now()->toIso8601ZuluString(),
            ],
        ]);
    }

    public function show(Request $request, SyncableEntityType $type)
    {
        // TODO: Review if this is how we want to handle this.
        $request->user()?->can('view', $type->getMorphedModel());

        $request->validate([
            'limit' => [
                'integer',
                'min:1',
                'max:1000',
            ],
            'since' => [
                'date',
                'nullable',
            ],
        ]);

        $entities = $type
            ->query($request->date('since'))
            // TODO: We need to be able to inject a custom query.
            // Probably a class which we inject here based on the type.
            // Ensuring users only have access to the data they should.
            ->orderBy('id', 'desc')
            ->cursorPaginate(
                perPage: $request->input('limit', 100)
            );

        $stateCollection = StateCollection::fromPaginator($entities);
        $transformedData = $stateCollection->transformToStateData();

        // Preserve the cursor paginator's internal state before replacing collection
        $nextCursor = $entities->nextCursor()?->encode();
        $prevCursor = $entities->previousCursor()?->encode();
        $hasMore = $entities->hasMorePages();

        // Build full URLs with limit parameter preserved
        $limit = $request->input('limit', 100);
        $baseUrl = $request->url();

        // Build query parameters for cursor URLs
        $queryParams = ['limit' => $limit];
        if ($request->has('since')) {
            $queryParams['since'] = $request->input('since');
        }

        $queryString = http_build_query($queryParams);

        $nextPageUrl = $nextCursor
            ? $baseUrl . '?cursor=' . $nextCursor . '&' . $queryString
            : null;

        $previousPageUrl = $prevCursor
            ? $baseUrl . '?cursor=' . $prevCursor . '&' . $queryString
            : null;

        // Build meta
        $meta = [
            'has_more' => $hasMore,
            'next_cursor' => $nextCursor,
            'prev_cursor' => $prevCursor,
            'entity_type' => $type->type,
            'server_timestamp' => now()->toIso8601ZuluString(),
        ];

        if ($request->date('since')) {
            $meta['deleted_sync_ids'] = $type
                ->getDeletedSince(
                    type: $type->type,
                    since: $request->date('since')
                )
                ->pluck('sync_id');
        }

        // Pass only the transformed collection, not the paginator
        return SyncableEntityResource::collection(
            resource: $transformedData
        )->additional([
            'meta' => $meta,
            'links' => [
                'first' => null,
                'last' => null,
                'prev' => $previousPageUrl,
                'next' => $nextPageUrl,
            ],
        ]);
    }
}
