<?php

declare(strict_types=1);

namespace CGA\Sync\Exceptions;

use CGA\Sync\Contracts\SyncableContract;
use CGA\Sync\Data\StateResolver;
use CGA\Sync\Values\ProcessedResult;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Response;

class SyncConflictException extends SyncException
{
    public function __construct(
        public private(set) string $syncId,
        public private(set) ?Model $conflictingModel = null,
        string $message = 'Revision conflict detected',
    ) {
        parent::__construct($message);
    }

    public static function revisionMismatch(string $syncId, ?Model $conflictingModel = null): self
    {
        return new self($syncId, $conflictingModel, __('Revision mismatch detected'));
    }

    private function getConflictData(): array
    {
        if (! $this->conflictingModel || ! is_subclass_of($this->conflictingModel, SyncableContract::class)) {
            return [
                'current_revision' => null,
                'current_state' => null,
            ];
        }

        $stateResolver = app(StateResolver::class);
        $currentState = $stateResolver->resolve($this->conflictingModel);

        return [
            'current_revision' => $currentState->getRevisionId(),
            'current_state' => $currentState->getFieldData(),
        ];
    }

    public function render(Request $request): Response|JsonResponse
    {
        if ($request->wantsJson()) {
            return response()->json(
                data: $this->toResultArray(),
                status: Response::HTTP_CONFLICT,
            );
        }

        return abort(Response::HTTP_CONFLICT, $this->getMessage());
    }

    public function toResultArray(): array
    {
        $conflictData = $this->getConflictData();

        return ProcessedResult::conflict(
            syncId: $this->syncId,
            message: $this->getMessage(),
            currentRevision: $conflictData['current_revision'],
            currentState: $conflictData['current_state'],
        )->toArray();
    }
}
