<?php

declare(strict_types=1);

namespace CGA\ScopeGroups\Models;

use CGA\ScopeGroups\Enums\Type;
use CGA\ScopeGroups\Events\ScopeGroupDeleting;
use CGA\ScopeGroups\Facades\ScopeGroups;
use Illuminate\Contracts\Auth\Authenticatable;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

class ScopeGroup extends Model
{
    /** @use HasFactory<\CGA\ScopeGroups\Database\Factories\ScopeGroupFactory> */
    use HasFactory;

    protected $guarded = [];

    protected $attributes = [
        'type' => Type::GLOBAL,
        'description' => '',
    ];

    /**
     * @return array<string, string>
     */
    protected function casts()
    {
        return [
            'type' => Type::class,
        ];
    }

    protected $dispatchesEvents = [
        'deleting' => ScopeGroupDeleting::class,
    ];

    public function __construct(array $attributes = [])
    {
        if (! isset($this->table)) {
            $this->setTable(config('scope-groups.table_names.scope_groups'));
        }

        parent::__construct($attributes);
    }

    /** @return HasMany<ScopeGroupMembership, $this> */
    public function memberships(): HasMany
    {
        return $this->hasMany(ScopeGroups::getScopeGroupMembershipModel());
    }

    /** @return HasMany<ScopeGroupResource, $this> */
    public function resources(): HasMany
    {
        return $this->hasMany(ScopeGroups::getScopeGroupResourceModel());
    }

    public function addMember(Model&Authenticatable $member): ScopeGroupMembership
    {
        return $this->memberships()->createOrFirst([
            'member_id' => $member->getKey(),
            'member_type' => $member->getMorphClass(),
        ]);
    }

    public function removeMember(Model&Authenticatable $member): void
    {
        $this->memberships()->where('member_id', $member->getKey())
            ->where('member_type', $member->getMorphClass())
            ->delete();
    }

    public function addResource(Model $resource): ScopeGroupResource
    {
        return $this->resources()->createOrFirst([
            'resource_id' => $resource->getKey(),
            'resource_type' => $resource->getMorphClass(),
        ]);
    }

    public function removeResource(Model $resource): void
    {
        $this->resources()->where('resource_id', $resource->getKey())
            ->where('resource_type', $resource->getMorphClass())
            ->delete();
    }

    public function addResources(iterable $resources): void
    {
        foreach ($resources as $resource) {
            $this->addResource($resource);
        }
    }
}
