<?php

declare(strict_types=1);

namespace CGA\ScopeGroups\AccessManagers;

use CGA\ScopeGroups\Contracts\IsScoped;
use CGA\ScopeGroups\Facades\ScopeGroups;
use Illuminate\Contracts\Auth\Authenticatable;

class Resolver
{
    protected array $cache = [];

    /**
     * @param  class-string  $model
     */
    public function resolve(string $model): BaseManager
    {
        if (isset($this->cache[$model])) {
            return $this->createManager($this->cache[$model], $model);
        }

        $handlerClass = $this->getHandlerClass($model);
        $this->cache[$model] = $handlerClass;

        return $this->createManager($handlerClass, $model);
    }

    protected function createManager(string $handlerClass, string $model): BaseManager
    {
        /** @var (IsScoped&Authenticatable)|null $user */
        $user = auth()->user();

        $contextClass = ScopeGroups::getScopeContextModel();
        $context = new $contextClass($user);

        return new $handlerClass($context, $model);
    }

    /**
     * @param  class-string  $model
     */
    protected function getHandlerClass(string $model)
    {
        $handlerNamespaces = $this->getNamespaces();
        $classBasename = class_basename($model);

        foreach ($handlerNamespaces as $namespace) {
            $handlerClass = $namespace . $classBasename . 'Manager';

            if (class_exists($handlerClass)) {
                return $handlerClass;
            }
        }

        return GenericManager::class;
    }

    protected function getNamespaces(): array
    {
        $configNamespaces = config('scope-groups.access_manager_namespaces', []);
        $defaultNamespaces = [
            'App\\Support\\ScopeGroups\\AccessManagers\\',
        ];

        // Config namespaces first (allow overrides), then defaults
        return array_merge($configNamespaces, $defaultNamespaces);
    }
}
