<?php

declare(strict_types=1);

namespace CGA\ScopeGroups\Services;

use CGA\ScopeGroups\Exceptions\ScopeGroupException;
use Illuminate\Support\Arr;
use Illuminate\Support\Str;
use ReflectionClass;
use Symfony\Component\Finder\Finder;

class ResolveModelsUsingTraits
{
    protected array $traits;

    public function __construct(
        array|string $trait
    ) {
        $this->traits = Arr::wrap($trait);
    }

    /**
     * @param string|string[] $namespaces
     */
    public function handle(string|array $namespaces): array
    {
        $namespaces = collect(Arr::wrap($namespaces))
            ->unique()
            ->map(fn(string $namespace) => rtrim($namespace, '\\') . '\\');

        if ($namespaces->isEmpty()) {
            throw new ScopeGroupException('No model namespaces configured');
        }

        $composer = require base_path('vendor/autoload.php');
        $psr4Prefixes = $composer->getPrefixesPsr4();

        $classes = [];

        foreach ($namespaces as $namespace) {
            $namespace = rtrim($namespace, '\\') . '\\';

            foreach ($psr4Prefixes as $prefix => $paths) {
                if (! Str::startsWith($namespace, $prefix)) {
                    continue;
                }

                // Calculate subdirectory path
                $relativeNamespace = substr($namespace, strlen($prefix));
                $subdirectory = str_replace('\\', DIRECTORY_SEPARATOR, $relativeNamespace);

                foreach ($paths as $basePath) {
                    $directory = rtrim($basePath, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . trim($subdirectory, DIRECTORY_SEPARATOR);

                    if (! is_dir($directory)) {
                        continue;
                    }

                    $foundClasses = $this->scanDirectory($directory, $namespace);
                    $classes = array_merge($classes, $foundClasses);
                }
            }
        }

        return array_unique($classes);
    }

    protected function scanDirectory(string $directory, string $baseNamespace): array
    {
        $classes = [];

        $finder = Finder::create()
            ->files()
            ->name('*.php')
            ->in($directory);

        foreach ($finder as $file) {
            $relativePath = $file->getRelativePathname();
            $class = $baseNamespace . str_replace(
                ['/', '.php'],
                ['\\', ''],
                $relativePath
            );

            if (class_exists($class) && $this->usesTraits($class)) {
                $classes[] = $class;
            }
        }

        return $classes;
    }

    protected function usesTraits(string $class): bool
    {
        try {
            $reflection = new ReflectionClass($class);

            if ($reflection->isAbstract() || $reflection->isInterface() || $reflection->isTrait()) {
                return false;
            }

            return ! empty(array_intersect($this->traits, class_uses_recursive($class)));
        } catch (\Throwable) {
            return false;
        }
    }
}
