<?php

declare(strict_types=1);

namespace CGA\ScopeGroups;

use CGA\ScopeGroups\Contracts\BoundaryExpander;
use CGA\ScopeGroups\Contracts\IsScoped;
use CGA\ScopeGroups\Exceptions\BoundaryExpanderException;
use CGA\ScopeGroups\Facades\ScopeGroups;
use Illuminate\Contracts\Auth\Authenticatable;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\Relation;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Config;

class ScopeContext
{
    protected array $expandedBoundaryCache = [];

    public function __construct(private (IsScoped&Authenticatable)|null $user = null) {}

    public function user(): (IsScoped&Authenticatable)|null
    {
        return $this->user;
    }

    public function scopeGroupIds(): Collection
    {
        return once(
            fn() => $this->user?->scopeGroups()
                ->select(app(ScopeGroups::getScopeGroupModel())->getTable() . '.id')
                ->pluck('id') ?? collect()
        );
    }

    /**
     * Get the boundary ids for a given model class
     *
     * @param  class-string<Model>  $modelClass  The model class to get the boundary ids for
     * @return Collection The boundary ids
     */
    public function boundaryIds(string $modelClass): Collection
    {
        $cacheKey = $modelClass;

        if (isset($this->expandedBoundaryCache[$cacheKey])) {
            return $this->expandedBoundaryCache[$cacheKey];
        }

        $directIds = once(
            fn() => ScopeGroups::getScopeGroupResourceModel()::query()
                ->whereIn('scope_group_id', $this->scopeGroupIds())
                ->where('resource_type', Relation::getMorphAlias($modelClass))
                ->pluck('resource_id')
                ->unique()
                ->values()
        );

        $expander = $this->getExpander($modelClass);
        if ($expander && $expander->handles($modelClass)) {
            $expandedIds = $expander->expand($modelClass, $directIds);
            $this->expandedBoundaryCache[$cacheKey] = $expandedIds;

            return $expandedIds;
        }

        $this->expandedBoundaryCache[$cacheKey] = $directIds;

        return $directIds;
    }

    protected function getExpander(string $modelClass): ?BoundaryExpander
    {
        $expanders = Config::get('scope-groups.boundary_expanders', []);

        if (isset($expanders[$modelClass])) {
            return $this->createExpander($expanders[$modelClass]);
        }

        if (isset($expanders['*'])) {
            return $this->createExpander($expanders['*']);
        }

        return null;
    }

    protected function createExpander(string $expanderClass): BoundaryExpander
    {
        $expander = app($expanderClass);

        if (! $expander instanceof BoundaryExpander) {
            throw BoundaryExpanderException::invalid($expanderClass);
        }

        return $expander;
    }

    /**
     * Check if the user has access to a given model class
     *
     * @param  class-string<Model>  $modelClass  The model class to check access for
     * @return bool True if the user has access, false otherwise
     */
    public function hasAccessTo(string $modelClass): bool
    {
        return $this->boundaryIds($modelClass)->isNotEmpty();
    }

    public function authorize(): bool
    {
        // If no user is authenticated, prevent access
        if (! $this->user) {
            return false;
        }

        return true;
    }

    public function shouldBypass(): bool
    {
        return false;
    }
}
