<?php

declare(strict_types=1);

namespace CGA\ScopeGroups\AccessManagers;

use CGA\ScopeGroups\ScopeContext;
use Illuminate\Contracts\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\ModelInspector;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Str;

abstract class BaseManager
{
    protected static array $inspectModelCache = [];

    /**
     * @param  class-string<Model>|null  $modelClass
     */
    public function __construct(
        protected ScopeContext $context,
        protected ?string $modelClass = null
    ) {
        //
    }

    public function before(): ?bool
    {
        if ($this->context->shouldBypass()) {
            return true; // Skip all scoping
        }

        if (! $this->context->authorize()) {
            return false; // No scope groups = deny all
        }

        return null; // Continue with normal scoping
    }

    public function applyScope(Builder $query): void
    {
        $boundaries = $this->getRelevantBoundaries();

        if (empty($boundaries)) {
            // Block any access.
            $query->whereRaw('1 = 0');

            return;
        }

        $query->where(function ($q) use ($boundaries) {
            foreach ($boundaries as $modelClass => $ids) {
                $this->applyBoundary($q, $modelClass, $ids);
            }
        });
    }

    /**
     * Get the model class this manager is responsible for.
     * Can be overridden to return a different model class.
     *
     * @return class-string<Model>
     */
    protected function getModelClass(): string
    {
        if ($this->modelClass !== null) {
            return $this->modelClass;
        }

        $className = class_basename(static::class);
        $modelName = str_replace('Manager', '', $className);

        return "App\\Models\\{$modelName}";
    }

    protected function getRelevantBoundaries(): array
    {
        $modelClass = $this->getModelClass();
        $hierarchy = $this->getHierarchy($modelClass);
        $boundaries = [];

        foreach ($hierarchy as $resourceModelClass) {
            $ids = $this->context->boundaryIds($resourceModelClass);
            if ($ids->isNotEmpty()) {
                $boundaries[$resourceModelClass] = $ids;
            }
        }

        return $boundaries;
    }

    /**
     * Apply a single boundary constraint to the query
     * Override this to customize logic for specific resource types
     *
     * @param  Builder  $query  The query builder to apply the boundary to
     * @param  class-string<Model>  $modelClass  The model class to apply the boundary to
     * @param  Collection  $ids  The ids to apply the boundary to
     */
    protected function applyBoundary(Builder $query, string $modelClass, Collection $ids): void
    {
        // Direct match (Teacher boundary → teacher.id)
        if ($modelClass === $this->getModelClass()) {
            $query->orWhereIn('id', $ids);

            return;
        }

        // Try convention-based approaches
        $this->applyBoundaryByConvention($query, $modelClass, $ids);
    }

    /**
     * Apply boundary using Laravel conventions (relationships, foreign keys)
     */
    protected function applyBoundaryByConvention(Builder $query, string $modelClass, Collection $ids): void
    {
        $model = $query->getModel();
        $inspectResult = $this->inspectModel($model::class);

        // Strategy 1: check for direct relationship
        $relation = $inspectResult['relations']->firstWhere('related', $modelClass);
        if ($relation) {
            $query->orWhereHas($relation['name'], fn($sq) => $sq->whereIn('id', $ids));

            return;
        }

        // Strategy 2: check for foreign key
        $attribute = $inspectResult['attributes']->firstWhere('name', "{$this->guessRelationShipName($modelClass)}_id");
        if ($attribute) {
            $query->orWhereIn($attribute['name'], $ids);

            return;
        }

        // Strategy 3: try nested relationship traversal
        $this->applyNestedBoundary($query, $modelClass, $ids);
    }

    protected function applyNestedBoundary(Builder $query, string $targetModelClass, Collection $ids): void
    {
        $modelClass = $this->getModelClass();
        $hierarchy = $this->getHierarchy($modelClass);
        $currentIndex = array_search($targetModelClass, $hierarchy);

        if ($currentIndex === false || $currentIndex === 0) {
            return;
        }

        // Build relationship path
        // If hierarchy is [Teacher, School, Geo] and target is Geo
        // We need: teacher.school.geo
        $relationshipPath = collect($hierarchy)
            ->take($currentIndex)
            ->map(function ($model, $index) use ($hierarchy) {
                $nextClass = $hierarchy[$index + 1];
                $inspectResult = $this->inspectModel($model);

                return $inspectResult['relations']->firstWhere('related', $nextClass)['name'];
            })
            ->filter()
            ->implode('.');

        if (empty($relationshipPath)) {
            return;
        }

        // Build nested whereHas
        $query->orWhereHas($relationshipPath, fn($sq) => $sq->whereIn('id', $ids));
    }

    protected function guessRelationShipName(string $modelClass): string
    {
        return Str::camel(class_basename($modelClass));
    }

    /**
     * Inspect the model and return the relationships and foreign keys
     *
     * @param  class-string<Model>  $modelClass
     * @return array<string, mixed>
     */
    protected function inspectModel(string $modelClass): array
    {
        if (isset(self::$inspectModelCache[$modelClass])) {
            return self::$inspectModelCache[$modelClass];
        }

        self::$inspectModelCache[$modelClass] = app(ModelInspector::class)->inspect($modelClass);

        return self::$inspectModelCache[$modelClass];
    }

    protected function getHierarchy(string $modelClass): array
    {
        $globalHierarchy = Config::get('scope-groups.global_hierarchy', []);
        $modelIndex = array_search($modelClass, $globalHierarchy);

        if ($modelIndex === false) {
            // Fallback to explicit config if model not in global hierarchy
            return Config::get("scope-groups.resource_hierarchy.{$modelClass}", [
                $modelClass, // Allow direct assignment
            ]);
        }

        // Return hierarchy from this model down to the root (including itself)
        return array_slice($globalHierarchy, $modelIndex);
    }
}
