<?php

declare(strict_types=1);

namespace CGA\ScopeGroups\Expanders;

use CGA\ScopeGroups\Contracts\BoundaryExpander;
use CGA\ScopeGroups\Exceptions\ScopeGroupException;
use CGA\ScopeGroups\Scopes\IsScoped;
use Illuminate\Support\Collection;
use Staudenmeir\LaravelAdjacencyList\Eloquent\HasRecursiveRelationships;

class HierarchicalBoundaryExpander implements BoundaryExpander
{
    public function expand(string $modelClass, Collection $ids): Collection
    {
        if (! trait_exists(HasRecursiveRelationships::class)) {
            throw new ScopeGroupException('The Staudenmeir\LaravelAdjacencyList package is not installed. Please install it to use this expander.');
        }

        if ($ids->isEmpty()) {
            return collect();
        }

        $model = app($modelClass);

        // Use treeOf to get all descendants of the given IDs in a single query
        // without loading individual models first
        $allDescendantIds = $model->newQuery()
            ->withoutGlobalScope(IsScoped::class) // Disable scope for this query
            ->treeOf(function ($query) use ($ids, $model) {
                $query->whereIn($model->getKeyName(), $ids);
            })
            ->pluck($model->getKeyName());

        return $ids->merge($allDescendantIds)->unique()->values();
    }

    public function handles(string $modelClass): bool
    {
        if (! class_exists($modelClass)) {
            return false;
        }

        $traits = class_uses_recursive($modelClass);

        return in_array(HasRecursiveRelationships::class, $traits, true);
    }
}
